using Inet.Viewer.Resources;
/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;

namespace Inet.Viewer.Data
{

    /// <summary>
    /// This class represents a range prompt value with one or two limits, each of which can be set to be included in the range or to be excluded from the range.
    /// @author LarsF
    /// </summary>
    public class RangePromptValue : PromptValue
    {

        private Range range;

        /// <summary>
        /// Creates an "empty" range prompt value with the given type </summary>
        /// <param name="type"> type of range prompt value </param>
        internal RangePromptValue(int type)
            : base(type)
        {
            range = new Range(null, null, false, false);
            InitValue();
        }

        /// <summary>
        /// The constructor. </summary>
        /// <param name="startValue"> The start value of the range. </param>
        /// <param name="endValue"> The end value of the range. </param>
        /// <param name="includeLow"> include the start value. </param>
        /// <param name="includeHigh"> include the end Value. </param>
        /// <param name="type"> value type of range. </param>
        internal RangePromptValue(SinglePromptValue startValue, SinglePromptValue endValue, bool includeLow, bool includeHigh, int type)
            : base("", type)
        {
            range = new Range(startValue, endValue, includeLow, includeHigh);
        }

        /// <summary>
        /// Constructor using a Range object </summary>
        /// <param name="range"> Range object to use to create a RangePromptValue based on the given range </param>
        /// <param name="type"> value type of the range </param>
        public RangePromptValue(Range range, int type)
            : base("", type)
        {
            this.range = range;
        }

        /// <returns> Is the upper limit included in the range? </returns>
        public bool IncludeHigh
        {
            get
            {
                return range.IncludeUpper;
            }
            set
            {
                range.IncludeUpper = value;
            }
        }


        /// <returns> Is the lower limit included in the range? </returns>
        public bool IncludeLow
        {
            get
            {
                return range.IncludeLower;
            }
            set
            {
                range.IncludeLower = value;
            }
        }


        /// <returns> Upper limit to the range, or null if there is no upper limit </returns>
        public virtual SinglePromptValue EndValue
        {
            get
            {
                return (SinglePromptValue)range.UpperValue;
            }
            set
            {
                range.UpperValue = value;
            }
        }


        /// <returns> Lower limit to the range, or null if there is no lower limit </returns>
        public virtual SinglePromptValue StartValue
        {
            get
            {
                return (SinglePromptValue)range.LowerValue;
            }
            set
            {
                range.LowerValue = value;
            }
        }


        // --------------------------------

        /// <summary>
        /// init the range values.
        /// </summary>
        internal override void InitValue()
        {
            if (range != null)
            {
                range.LowerValue = new SinglePromptValue(this.Type);
                range.UpperValue = new SinglePromptValue(this.Type);
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override object Value
        {
            get
            {
                return this;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        internal override string ValueString
        {
            get
            {
                string resultString = range.LowerValue.ToString();
                resultString += " ";
                if (!IncludeLow)
                {
                    resultString += "("+strings.Prompt_Excl+") ";
                }
                resultString += strings.Prompt_To;
                resultString += " ";
                resultString += range.UpperValue.ToString();
                if (!IncludeHigh)
                {
                    resultString += " (" + strings.Prompt_Excl + ")";
                }
                return resultString;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// Returns the i-net Clear Reports formula syntax representation of this range. 
        /// </summary>
        public override string StringRepresentation
        {
            get
            {
                if (StartValue.Value == null && EndValue.Value == null)
                {
                    return "null";
                }
                return StartValue.StringRepresentation + " " + (IncludeLow ? "" : "_") + "to" + (IncludeHigh ? "" : "_") + " " + EndValue.StringRepresentation;

            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override bool Equals(object obj)
        {
            if (obj is RangePromptValue)
            {
                RangePromptValue value = (RangePromptValue)obj;
                return value.IncludeHigh == IncludeHigh && value.IncludeLow == IncludeLow && value.EndValue.Equals(EndValue) && value.StartValue.Equals(StartValue) && value.Description.Equals(Description) && value.OnlyDescription == OnlyDescription;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override int GetHashCode()
        {
            string idString = Description + OnlyDescription + StartValue + EndValue + IncludeLow + IncludeHigh;
            return idString.GetHashCode();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public string CheckRange()
        {
            object l = range.LowerValue;
            if (l is SinglePromptValue)
            {
                l = ((SinglePromptValue)l).Value;
            }
            object r = range.UpperValue;
            if (r is SinglePromptValue)
            {
                r = ((SinglePromptValue)r).Value;
            }
            if (l != null && r != null)
            {

                switch (this.Type)
                {
                    case PromptData.Boolean:
                    case PromptData.Binary:
                        return null;
                    case PromptData.Currency:
                    case PromptData.Number:
                        if (((double)l).CompareTo((double)r) <= 0)
                        {
                            return null;
                        }
                        break;
                    case PromptData.Date:
                    case PromptData.Datetime:
                        DateTime left = new DateTime();
                        if (l is DateTime)
                        {
                            left = (DateTime)l;
                        }
                        else
                        {
                            return "Invalid value: " + l;
                        }
                        if (left.CompareTo(r) <= 0)
                        {
                            return null;
                        }
                        break;
                    case PromptData.Time:
                        if (((TimeSpan)l).CompareTo((TimeSpan)r) <= 0)
                        {
                            return null;
                        }
                        break;
                    case PromptData.String:
                        if (((string)l).CompareTo((string)r) <= 0)
                        {

                            return null;
                        }
                        break;
                }
            }
            else if (l == null && r == null)
            {
                return null;
            }
            return strings.Prompt_Error_InvalidRange;
        }

    }

}